#!/bin/bash
# =============================================================================
# wg_handy_setup.sh
# Erzeugt WireGuard-Schlüsselpaar, handy.conf, Peer-Eintrag für wg0.conf
# und QR-Code als PNG für WG-Tunnel (Android)
# =============================================================================

set -e

# Arbeitsverzeichnis auf den Ordner des Skripts setzen
cd "$(dirname "$0")"

echo "=============================================="
echo " WireGuard Handy-Setup für WG-Tunnel (Android)"
echo "=============================================="
echo ""

# --- Abhängigkeiten prüfen und installieren ---
echo "Prüfe ob wireguard-tools und qrencode installiert sind..."
PACKAGES=""
if ! command -v wg &> /dev/null; then
    PACKAGES="$PACKAGES wireguard-tools"
fi
if ! command -v qrencode &> /dev/null; then
    PACKAGES="$PACKAGES qrencode"
fi

if [ -n "$PACKAGES" ]; then
    echo "Installiere fehlende Pakete: $PACKAGES"
    sudo apt install -y $PACKAGES
else
    echo "Alle benötigten Pakete sind bereits installiert."
fi
echo ""

# --- Eingaben abfragen ---
read -p "Tunnel-IP des Handys      (z.B. 10.99.99.5):   " HANDY_IP
read -p "Tunnel-IP des Raspis      (z.B. 10.99.99.2):   " RASPI_IP
read -p "Public Key des Raspis:                          " RASPI_PUBKEY
read -p "DynDNS oder öffentliche IP des Raspis:          " RASPI_ENDPOINT
read -p "WireGuard-Port            (Standard: 51820):    " WG_PORT
WG_PORT=${WG_PORT:-51820}
echo ""

# --- Schlüsselpaar für das Handy erzeugen ---
echo "Erzeuge Schlüsselpaar für das Handy..."
HANDY_PRIVKEY=$(wg genkey)
HANDY_PUBKEY=$(echo "$HANDY_PRIVKEY" | wg pubkey)
echo "  Private Key: $HANDY_PRIVKEY"
echo "  Public Key:  $HANDY_PUBKEY"
echo ""

# --- handy.conf erzeugen ---
echo "Erzeuge handy.conf..."
cat > handy.conf << EOF
[Interface]
PrivateKey = ${HANDY_PRIVKEY}
Address = ${HANDY_IP}/24

[Peer]
PublicKey = ${RASPI_PUBKEY}
AllowedIPs = ${RASPI_IP}/32
Endpoint = ${RASPI_ENDPOINT}:${WG_PORT}
PersistentKeepalive = 25
EOF
echo "  handy.conf erstellt."
echo ""

# --- Peer-Eintrag für wg0.conf des Raspis erzeugen ---
echo "Erzeuge peer_eintrag_fuer_wg0conf.txt..."
cat > peer_eintrag_fuer_wg0conf.txt << EOF
# Diesen Block in die wg0.conf auf dem Raspi einfügen:
# sudo nano /etc/wireguard/wg0.conf

# Handy (WG-Tunnel Android)
[Peer]
PublicKey = ${HANDY_PUBKEY}
AllowedIPs = ${HANDY_IP}/32
PersistentKeepalive = 25

# Danach WireGuard neu laden:
# sudo wg syncconf wg0 <(wg-quick strip wg0)
EOF
echo "  peer_eintrag_fuer_wg0conf.txt erstellt."
echo ""

# --- Textdatei zur Kontrolle erzeugen ---
echo "Erzeuge handy_config_kontrolle.txt..."
cp handy.conf handy_config_kontrolle.txt
echo "  handy_config_kontrolle.txt erstellt."
echo "  ACHTUNG: Enthält den Private Key – nach Kontrolle selbst löschen!"
echo ""

# --- QR-Code als PNG erzeugen ---
echo "Erzeuge QR-Code als PNG..."
qrencode -t png -o handy_qrcode.png < handy.conf
echo "  handy_qrcode.png erstellt."
echo ""

# --- Zusammenfassung ---
echo "=============================================="
echo " Zusammenfassung"
echo "=============================================="
echo ""
echo "1. Öffne handy_qrcode.png auf deinem Bildschirm"
echo "   und scanne ihn mit WG-Tunnel (Android)."
echo ""
echo "2. Trage den folgenden Peer-Block in die"
echo "   wg0.conf auf deinem Raspi ein:"
echo ""
cat peer_eintrag_fuer_wg0conf.txt
echo ""
echo "3. In Sipnetic den SIP-Account eintragen:"
echo "   SIP-Server: ${RASPI_IP}"
echo "   TLS/SRTP:   deaktiviert"
echo ""

# --- Temporäre Dateien löschen ---
echo "Lösche handy.conf (enthält den Private Key)..."
rm -f handy.conf
echo "  Erledigt."
echo ""
echo "WICHTIG: Nach dem Scannen und nach dem Eintragen"
echo "         auf dem Raspi diese Dateien löschen:"
echo "         rm handy_qrcode.png peer_eintrag_fuer_wg0conf.txt handy_config_kontrolle.txt"
echo ""
echo "=============================================="
echo " Fertig!"
echo "=============================================="
